// ========= Fiorela Linktree (oscuro delicado + logos por archivo + toast verde) =========
const config = {
  displayName: "Fiorela",
  bio: "Creadora de contenido | Belleza & lifestyle ✨",
  theme: "light", // 'light' o 'dark'

  // REDES: IG + 2 TikTok (iconos SVG inline, no imágenes)
  socials: [
    { name: "Instagram", url: "https://www.instagram.com/fiorejd28/", icon: "instagram" },
    { name: "TikTok (principal)", url: "https://www.tiktok.com/@fiorejd", icon: "tiktok" },
    { name: "TikTok (secundario)", url: "https://www.tiktok.com/@fiorejd22", icon: "tiktok" }
  ],

  // 1) PORTAFOLIO (solo)
  portfolioLinks: [
    { type: "link", title: "Portfolio", subtitle: "Diseño de imagen y sonido — proyectos", url: "https://fiorejd28.my.canva.site/", logoSrc: "assets/icons/porfolio.jpg" }
  ],

  // 2) YESSTYLE
  yesstyleLinks: [
    { type: "link", title: "Mis favoritos en Yesstyle", subtitle: "Mi selección de productos", url: "#", logoSrc: "assets/icons/yesstile.png" },
    { type: "link", title: "¿Quieres se influencer en Yesstyle?", subtitle: "Únete ahora (enlace oficial)", url: "https://www.yesstyle.com/es/influencers.html?irirco=FIOREJD28&utm_term=FIOREJD28&utm_medium=infreferral&mcg=influencer&utm_source=influencer&utm_campaign=iriprivate", logoSrc: "assets/icons/yesstile.png" }
  ],

  // 3) CÓDIGOS DE DESCUENTO
  discountLinks: [
    { type: "copy", title: "Yesstyle FIOREJD28",       subtitle: "Código de descuento", code: "FIOREJD28", logoSrc: "assets/icons/yesstile.png" },
    { type: "copy", title: "Flower Knows FIOREJD28",   subtitle: "Código de descuento", code: "FIOREJD28", logoSrc: "assets/icons/flower.jpg" },
    { type: "copy", title: "StyleKorean FIOREJD28",    subtitle: "Código de descuento", code: "FIOREJD28", logoSrc: "assets/icons/korean.jpg" }
  ]
};

/* ====== ICONOS SVG DE REDES (inline) ====== */
const svgs = {
  instagram: `<svg viewBox="0 0 24 24" fill="none" aria-hidden="true"><rect x="2" y="2" width="20" height="20" rx="5" stroke="currentColor" stroke-width="1.8"/><path d="M12 8.2a3.8 3.8 0 1 0 0 7.6 3.8 3.8 0 0 0 0-7.6Z" stroke="currentColor" stroke-width="1.8"/><circle cx="17.4" cy="6.6" r="1.2" fill="currentColor"/></svg>`,
  tiktok:    `<svg viewBox="0 0 24 24" fill="currentColor" aria-hidden="true"><path d="M14.5 3.5c.7 2.4 2.3 4 4.8 4.3v3.1c-1.9.1-3.6-.6-4.8-1.7v5.8a6.2 6.2 0 1 1-6.2-6.2c.5 0 1 .1 1.5.2v3.3a2.9 2.9 0 1 0 2.9 2.9V3.5h1.8Z"/></svg>`
};

/* ====== Helpers ====== */
const $ = (sel, root=document) => root.querySelector(sel);

function iconSpan(name, cls="ico"){
  const span = document.createElement("span");
  span.className = cls;
  span.innerHTML = svgs[name] || "🔗";
  return span;
}

function showToast(message){
  const toast = $("#toast");
  toast.textContent = message;
  toast.classList.add("show");
  setTimeout(()=> toast.classList.remove("show"), 1400);
}

function addRipple(e, el){
  const r = document.createElement("span");
  r.className = "ripple";
  const rect = el.getBoundingClientRect();
  const size = Math.max(rect.width, rect.height);
  r.style.width = r.style.height = size + "px";
  r.style.left = (e.clientX - rect.left - size/2) + "px";
  r.style.top = (e.clientY - rect.top - size/2) + "px";
  el.appendChild(r);
  setTimeout(()=> r.remove(), 600);
}

/* ====== Render ====== */
function renderHeader(){
  $("#displayName").textContent = config.displayName;
  $("#bio").textContent = config.bio;
  $("#year").textContent = new Date().getFullYear();

  if (config.theme === "light") document.documentElement.classList.remove("dark");
  else document.documentElement.classList.add("dark");

  const socialsNav = $(".socials");
  socialsNav.innerHTML = "";
  config.socials.forEach(s => {
    const a = document.createElement("a");
    a.className = "social-btn";
    a.href = s.url;
    a.target = "_blank";
    a.rel = "noopener";
    a.setAttribute("aria-label", s.name);
    a.appendChild(iconSpan(s.icon));
    socialsNav.appendChild(a);
  });
}

/* Render genérico de una lista en un contenedor */
function renderSection(list, mountId){
  const mount = document.getElementById(mountId);
  mount.innerHTML = "";
  const tpl = document.getElementById("link-card");

  list.forEach(item => {
    const node = tpl.content.cloneNode(true);
    const a = node.querySelector("a.card");
    const titleEl = node.querySelector(".title");
    const subEl = node.querySelector(".subtitle");
    const chipEl = node.querySelector(".chip");
    const logoBox = node.querySelector(".icon.icon-logo");

    titleEl.textContent = item.title;
    subEl.textContent = item.subtitle || "";

    // LOGO por archivo (SVG/PNG/JPG). Si falta, usa iniciales del título.
    if (item.logoSrc) {
      const img = document.createElement("img");
      img.src = item.logoSrc;
      img.alt = (item.title || "logo");
      img.loading = "lazy";
      img.decoding = "async";
      logoBox.innerHTML = "";
      logoBox.appendChild(img);
    } else {
      logoBox.textContent = (item.title || "?").slice(0,2).toUpperCase();
    }

    if (item.type === "copy") {
      a.classList.add("copy");
      a.href = "#";
      a.setAttribute("role", "button");
      a.setAttribute("aria-label", `${item.title}: copiar código`);
      chipEl.textContent = "COPIAR CÓDIGO";
      a.addEventListener("click", async (e) => {
        e.preventDefault();
        addRipple(e, a);
        try {
          await navigator.clipboard.writeText(item.code);
          const prev = subEl.textContent;
          subEl.textContent = "✓ ¡Copiado!";
          subEl.classList.add("copied");
          showToast("✓ Código copiado");
          setTimeout(() => {
            subEl.textContent = prev;
            subEl.classList.remove("copied");
          }, 1400);
        } catch {
          alert("No se pudo copiar el código 😅");
        }
      });
    } else {
      a.classList.add("link");
      a.href = item.url;
      a.setAttribute("aria-label", `${item.title}: abrir enlace`);
      a.addEventListener("click", (e) => addRipple(e, a));
    }

    mount.appendChild(node);
  });
}

function renderAll(){
  renderHeader();
  renderSection(config.portfolioLinks, "links-portfolio"); // 1) Portafolio
  renderSection(config.yesstyleLinks, "links-yesstyle");   // 2) Yesstyle
  renderSection(config.discountLinks, "links-codes");      // 3) Códigos
}

/* ====== Setup ====== */
function setupTheme(){
  const toggle = document.getElementById("themeToggle");
  toggle.addEventListener("click", ()=>{
    document.documentElement.classList.toggle("dark");
  });
}

// Fondo decorativo (canvas)
function drawDecor(){
  const c = document.getElementById("bg-ornaments");
  const ctx = c.getContext("2d");

  function resize(){
    c.width = window.innerWidth;
    c.height = window.innerHeight;
  }
  resize();
  window.addEventListener("resize", resize);

  const blobs = Array.from({length: 8}).map(()=> ({
    x: Math.random()*c.width,
    y: Math.random()*c.height,
    r: 80 + Math.random()*120,
    dx: (Math.random()-.5)*0.4,
    dy: (Math.random()-.5)*0.4,
    hue: Math.random()<.5 ? 345 : 15
  }));

  function tick(){
    ctx.clearRect(0,0,c.width,c.height);
    blobs.forEach(b=>{
      ctx.beginPath();
      const grad = ctx.createRadialGradient(b.x, b.y, 0, b.x, b.y, b.r);
      const color1 = `hsla(${b.hue}, 92%, 74%, .18)`;
      const color2 = `hsla(${b.hue}, 92%, 74%, 0)`;
      grad.addColorStop(0, color1);
      grad.addColorStop(1, color2);
      ctx.fillStyle = grad;
      ctx.arc(b.x, b.y, b.r, 0, Math.PI*2);
      ctx.fill();

      b.x += b.dx; b.y += b.dy;
      if(b.x < -b.r || b.x > c.width + b.r) b.dx *= -1;
      if(b.y < -b.r || b.y > c.height + b.r) b.dy *= -1;
    });
    requestAnimationFrame(tick);
  }
  if (!window.matchMedia("(prefers-reduced-motion: reduce)").matches){
    tick();
  }
}

/* ====== Init ====== */
document.addEventListener("DOMContentLoaded", ()=>{
  renderAll();
  setupTheme();
  drawDecor();
  document.getElementById("year").textContent = new Date().getFullYear();
});
